/* SPDX-License-Identifier: GPL-2.0-or-later */

/*
 * Copyright (C) 2015 by pierrr kuo <vichy.kuo@gmail.com>
 */

#ifndef OPENOCD_TARGET_ARMV8_OPCODES_H
#define OPENOCD_TARGET_ARMV8_OPCODES_H

#include "arm_opcodes.h"

#define SYSTEM_CUREL_MASK		0xC0
#define SYSTEM_CUREL_SHIFT		6
#define SYSTEM_CUREL_EL0		0x0
#define SYSTEM_CUREL_EL1		0x1
#define SYSTEM_CUREL_EL2		0x2
#define SYSTEM_CUREL_EL3		0x3
#define SYSTEM_CUREL_NONCH		0xF
#define SYSTEM_AARCH64			0x1

#define SYSTEM_AAR64_MODE_EL0T	0x0
#define SYSTEM_AAR64_MODE_EL1T	0x4
#define SYSTEM_AAR64_MODE_EL1H	0x5
#define SYSTEM_AAR64_MODE_EL2T	0x8
#define SYSTEM_AAR64_MODE_EL2H	0x9
#define SYSTEM_AAR64_MODE_EL3T	0xC
#define SYSTEM_AAR64_MODE_EL3H	0xd

#define SYSTEM_DAIF			0xDA11
#define SYSTEM_DAIF_MASK		0x3C0
#define SYSTEM_DAIF_SHIFT		6

#define SYSTEM_ELR_EL1			0xC201
#define SYSTEM_ELR_EL2			0xE201
#define SYSTEM_ELR_EL3			0xF201

#define SYSTEM_SCTLR_EL1	0xC080
#define SYSTEM_SCTLR_EL2	0xE080
#define SYSTEM_SCTLR_EL3	0xF080

#define SYSTEM_FPCR		0xDA20
#define SYSTEM_FPSR		0xDA21
#define SYSTEM_DAIF		0xDA11
#define SYSTEM_NZCV		0xDA10
#define SYSTEM_SP_EL0		0xC208
#define SYSTEM_SP_EL1		0xE208
#define SYSTEM_SP_EL2		0xF208
#define SYSTEM_SP_SEL		0xC210
#define SYSTEM_SPSR_ABT		0xE219
#define SYSTEM_SPSR_FIQ		0xE21B
#define SYSTEM_SPSR_IRQ		0xE218
#define SYSTEM_SPSR_UND		0xE21A

#define SYSTEM_SPSR_EL1			0xC200
#define SYSTEM_SPSR_EL2			0xE200
#define SYSTEM_SPSR_EL3			0xF200

#define SYSTEM_ISR_EL1			0xC608

#define SYSTEM_DBG_DSPSR_EL0    0xDA28
#define SYSTEM_DBG_DLR_EL0		0xDA29
#define SYSTEM_DBG_DTRRX_EL0	0x9828
#define SYSTEM_DBG_DTRTX_EL0	0x9828
#define SYSTEM_DBG_DBGDTR_EL0	0x9820

#define SYSTEM_CCSIDR			0xC800
#define SYSTEM_CLIDR			0xC801
#define SYSTEM_CSSELR			0xD000
#define SYSTEM_CTYPE			0xD801
#define SYSTEM_CTR			0xD801

#define SYSTEM_DCCISW			0x43F2
#define SYSTEM_DCCSW			0x43D2
#define SYSTEM_ICIVAU			0x5BA9
#define SYSTEM_DCCVAU			0x5BD9
#define SYSTEM_DCCIVAC			0x5BF1

#define SYSTEM_MPIDR			0xC005

#define SYSTEM_AMAIR_EL1		0b1100010100011000
#define SYSTEM_AMAIR_EL2		0b1110010100011000
#define SYSTEM_AMAIR_EL3		0b1111010100011000

#define SYSTEM_TCR_EL1			0xC102
#define SYSTEM_TCR_EL2			0xE102
#define SYSTEM_TCR_EL3			0xF102

#define SYSTEM_TTBR0_EL1		0xC100
#define SYSTEM_TTBR0_EL2		0xE100
#define SYSTEM_TTBR0_EL3		0xF100

#define SYSTEM_MAIR_EL1			0b1100010100010000
#define SYSTEM_MAIR_EL2			0b1110010100010000
#define SYSTEM_MAIR_EL3			0b1111010100010000

#define SYSTEM_RMR_EL1			0b1100011000000010
#define SYSTEM_RMR_EL2			0b1110011000000010
#define SYSTEM_RMR_EL3			0b1111011000000010

#define SYSTEM_RVBAR_EL3		0b1111011000000001
#define SYSTEM_SDER32_EL3		0b1111000010001001

#define SYSTEM_TTBR1_EL1		0xC101
#define SYSTEM_TTBR1_EL2		0b1110000100000001

#define SYSTEM_ACTLR_EL1		0b1100000010000001
#define SYSTEM_ACTLR_EL2		0b1110000010000001
#define SYSTEM_ACTLR_EL3		0b1111000010000001

#define SYSTEM_TPIDRRO_EL0		0b1101111010000011

#define SYSTEM_TPIDR_EL0		0b1101111010000010
#define SYSTEM_TPIDR_EL1		0b1100011010000100
#define SYSTEM_TPIDR_EL2		0b1110011010000010
#define SYSTEM_TPIDR_EL3		0b1111011010000010

#define SYSTEM_AFSR0_EL1		0b1100001010001000
#define SYSTEM_AFSR0_EL2		0b1110001010001000
#define SYSTEM_AFSR0_EL3		0b1111001010001000
#define SYSTEM_AFSR1_EL1		0b1100001010001001
#define SYSTEM_AFSR1_EL2		0b1110001010001001
#define SYSTEM_AFSR1_EL3		0b1111001010001001

#define SYSTEM_CPTR_EL2			0b1110000010001010
#define SYSTEM_CPTR_EL3			0b1111000010001010

#define SYSTEM_CONTEXTIDR_EL1	0b1100011010000001
#define SYSTEM_CONTEXTIDR_EL2	0b1110011010000001

#define SYSTEM_DISR_EL1			0b1100011000001001
#define SYSTEM_HPFAR_EL2		0b1110001100000100
#define SYSTEM_IFSR32_EL2		0b1110001010000001

#define SYSTEM_DACR32_EL2		0b1110000110000000
#define SYSTEM_HACR_EL2			0b1110000010001111
#define SYSTEM_HSTR_EL2			0b1110000010001011

#define SYSTEM_CNTFRQ_EL0		0b1101111100000000
#define SYSTEM_CNTPCT_EL0		0b1101111100000001
#define SYSTEM_CNTVCT_EL0		0b1101111100000010
#define SYSTEM_CNTP_TVAL_EL0	0b1101111100010000
#define SYSTEM_CNTP_CTL_EL0		0b1101111100010001
#define SYSTEM_CNTP_CVAL_EL0	0b1101111100010010
#define SYSTEM_CNTV_TVAL_EL0	0b1101111100011000
#define SYSTEM_CNTV_CTL_EL0		0b1101111100011001
#define SYSTEM_CNTV_CVAL_EL0	0b1101111100011010

#define SYSTEM_CNTKCTL_EL1		0b1100011100001000
#define SYSTEM_CNTPS_TVAL_EL1	0b1111111100010000
#define SYSTEM_CNTPS_CTL_EL1	0b1111111100010001
#define SYSTEM_CNTPS_CVAL_EL1	0b1111111100010010

#define SYSTEM_CNTVOFF_EL2		0b1110011100000011
#define SYSTEM_CNTHCTL_EL2		0b1110011100001000
#define SYSTEM_CNTHP_TVAL_EL2	0b1110011100010000
#define SYSTEM_CNTHP_CTL_EL2	0b1110011100010000
#define SYSTEM_CNTHP_CVAL_EL2	0b1110011100010010
#define SYSTEM_CNTHV_TVAL_EL2	0b1110011100011000
#define SYSTEM_CNTHV_CTL_EL2	0b1110011100011001
#define SYSTEM_CNTHV_CVAL_EL2	0b1110011100011010

#define SYSTEM_ERRIDR_EL1       0b1100001010011000
#define SYSTEM_ERRSELR_EL1      0b1100001010011001
#define SYSTEM_ERXADDR_EL1      0b1100001010100011
#define SYSTEM_ERXCTLR_EL1      0b1100001010100001
#define SYSTEM_ERXFR_EL1        0b1100001010100000
#define SYSTEM_ERXMISC0_EL1     0b1100001010101000
#define SYSTEM_ERXMISC1_EL1     0b1100001010101001
#define SYSTEM_ERXSTATUS_EL1    0b1100001010100010

#define SYSTEM_REVIDR_EL1       0b1100000000000110

#define SYSTEM_CPACR_EL1		0b1100000010000010
#define SYSTEM_CSSELR_EL1		0b1101000000000000
#define SYSTEM_HCR_EL2			0b1110000010001000
#define SYSTEM_SCR_EL3			0b1111000010001000
#define SYSTEM_VTCR_EL2			0b1110000100001010
#define SYSTEM_VTTBR_EL2		0b1110000100001000

#define SYSTEM_VDISR_EL2		0b1110011000001001
#define SYSTEM_VSESR_EL2		0b1110001010010011

#define SYSTEM_VMPIDR_EL2       0b1110000000000101
#define SYSTEM_VPIDR_EL2        0b1110000000000000

#define SYSTEM_CPUCFR_EL1		0b1100011110000000
#define SYSTEM_CPUPWRCTLR_EL1	0b1100011110010111
#define SYSTEM_CCSIDR_EL1		0b1100100000000000
#define SYSTEM_CLIDR_EL1		0b1100100000000001

#define SYSTEM_DBGAUTHSTATUS_EL1 0b1000001111110110
#define SYSTEM_DBGCLAIMCLR_EL1	 0b1000001111001110
#define SYSTEM_DBGCLAIMSET_EL1   0b1000001111000110
#define SYSTEM_DBGDTRRX_EL0		 0b1001100000101000
#define SYSTEM_DBGDTRTX_EL0      0b1001100000101000
#define SYSTEM_DBGDTR_EL0		 0b1001100000100000
#define SYSTEM_DBGPRCR_EL1		 0b1000000010100100
#define SYSTEM_DBGVCR32_EL2      0b1010000000111000
#define SYSTEM_DBGWVR0_EL1       0b1000000000000110
#define SYSTEM_DBGWVR1_EL1       0b1000000000001110
#define SYSTEM_DBGWVR2_EL1		 0b1000000000010110
#define SYSTEM_DBGWVR3_EL1       0b1000000000011110
#define SYSTEM_DBGWCR0_EL1		 0b1000000000000111
#define SYSTEM_DBGWCR1_EL1		 0b1000000000001111
#define SYSTEM_DBGWCR2_EL1	     0b1000000000010111
#define SYSTEM_DBGWCR3_EL1		 0b1000000000011111
#define SYSTEM_DCZID_EL0         0b1101100000000111

#define SYSTEM_ICC_AP0R0_EL1	0b1100011001000100
#define SYSTEM_ICC_AP1R0_EL1	0b1100011001001000
#define SYSTEM_ICC_ASGI1R_EL1	0b1100011001011110
#define SYSTEM_ICC_BPR0_EL1		0b1100011001000011
#define SYSTEM_ICC_BPR1_EL1		0b1100011001100011
#define SYSTEM_ICC_CTLR_EL1		0b1100011001100100
#define SYSTEM_ICC_DIR_EL1		0b1100011001011001
#define SYSTEM_ICC_EOIR0_EL1	0b1100011001000001
#define SYSTEM_ICC_EOIR1_EL1	0b1100011001100001
#define SYSTEM_ICC_HPPIR0_EL1	0b1100011001000010
#define SYSTEM_ICC_HPPIR1_EL1	0b1100011001100010
#define SYSTEM_ICC_IAR0_EL1		0b1100011001000000
#define SYSTEM_ICC_IAR1_EL1		0b1100011001100000
#define SYSTEM_ICC_IGRPEN0_EL1	0b1100011001100110
#define SYSTEM_ICC_IGRPEN1_EL1	0b1100011001100111
#define SYSTEM_ICC_PMR_EL1		0b1100001000110000
#define SYSTEM_ICC_RPR_EL1		0b1100011001011011
#define SYSTEM_ICC_SGI0R_EL1	0b1100011001011111
#define SYSTEM_ICC_SGI1R_EL1	0b1100011001011101
#define SYSTEM_ICC_SRE_EL1		0b1100011001100101
#define SYSTEM_ICV_AP0R0_EL1	0b1100011001000100
#define SYSTEM_ICV_AP1R0_EL1	0b1100011001001000
#define SYSTEM_ICV_BPR0_EL1		0b1100011001000011
#define SYSTEM_ICV_BPR1_EL1		0b1100011001100011
#define SYSTEM_ICV_CTLR_EL1		0b1100011001100100
#define SYSTEM_ICV_DIR_EL1		0b1100011001011001
#define SYSTEM_ICV_EOIR0_EL1	0b1100011001000001
#define SYSTEM_ICV_EOIR1_EL1	0b1100011001100001
#define SYSTEM_ICV_HPPIR0_EL1	0b1100011001000010
#define SYSTEM_ICV_HPPIR1_EL1	0b1100011001100010
#define SYSTEM_ICV_IAR0_EL1		0b1100011001000000
#define SYSTEM_ICV_IAR1_EL1		0b1100011001100000
#define SYSTEM_ICV_IGRPEN0_EL1	0b1100011001100110
#define SYSTEM_ICV_IGRPEN1_EL1	0b1100011001100111
#define SYSTEM_ICV_PMR_EL1		0b1100001000110000
#define SYSTEM_ICV_RPR_EL1		0b1100011001011011

#define SYSTEM_ICH_AP0R0_EL2	0b1110011001000000
#define SYSTEM_ICH_AP1R0_EL2	0b1110011001001000
#define SYSTEM_ICH_EISR_EL2		0b1110011001011011
#define SYSTEM_ICH_ELRSR_EL2	0b1110011001011101
#define SYSTEM_ICH_HCR_EL2		0b1110011001011000
#define SYSTEM_ICH_LR0_EL2		0b1110011001100000
#define SYSTEM_ICH_LR1_EL2		0b1110011001100001
#define SYSTEM_ICH_LR2_EL2		0b1110011001100010
#define SYSTEM_ICH_LR3_EL2		0b1110011001100011
#define SYSTEM_ICH_MISR_EL2		0b1110011001011010
#define SYSTEM_ICH_VMCR_EL2		0b1110011001011111
#define SYSTEM_ICH_VTR_EL2		0b1110011001011001
#define SYSTEM_ICC_SRE_EL2		0b1110011001001101

#define SYSTEM_ICC_CTLR_EL3		0b1111011001100100
#define SYSTEM_ICC_IGRPEN1_EL3	0b1111011001100111
#define SYSTEM_ICC_SRE_EL3		0b1111011001100101

#define SYSTEM_ID_AA64AFR0_EL1	0b1100000000101100
#define SYSTEM_ID_AA64AFR1_EL1	0b1100000000101101
#define SYSTEM_ID_AA64DFR0_EL1	0b1100000000101000
#define SYSTEM_ID_AA64DFR1_EL1	0b1100000000101001
#define SYSTEM_ID_AA64ISAR0_EL1	0b1100000000110000
#define SYSTEM_ID_AA64ISAR1_EL1	0b1100000000110001
#define SYSTEM_ID_AA64MMFR0_EL1	0b1100000000111000
#define SYSTEM_ID_AA64MMFR1_EL1	0b1100000000111001
#define SYSTEM_ID_AA64MMFR2_EL1	0b1100000000111010
#define SYSTEM_ID_AA64PFR0_EL1	0b1100000000100000
#define SYSTEM_ID_AA64PFR1_EL1	0b1100000000100001
#define SYSTEM_ID_AFR0_EL1		0b1100000000001011
#define SYSTEM_ID_ISAR0_EL1		0b1100000000010000
#define SYSTEM_ID_ISAR1_EL1		0b1100000000010001
#define SYSTEM_ID_ISAR2_EL1		0b1100000000010010
#define SYSTEM_ID_ISAR3_EL1		0b1100000000010011
#define SYSTEM_ID_ISAR4_EL1		0b1100000000010100
#define SYSTEM_ID_ISAR5_EL1		0b1100000000010101
#define SYSTEM_ID_MMFR0_EL1		0b1100000000001100
#define SYSTEM_ID_MMFR1_EL1		0b1100000000001101
#define SYSTEM_ID_MMFR2_EL1		0b1100000000001110
#define SYSTEM_ID_MMFR3_EL1		0b1100000000001111
#define SYSTEM_ID_MMFR4_EL1		0b1100000000010110
#define SYSTEM_ID_PFR0_EL1		0b1100000000001000
#define SYSTEM_ID_PFR1_EL1		0b1100000000001001

#define SYSTEM_LORID_EL1		0b1100010100100111

#define SYSTEM_MDCCINT_EL1		0b1000000000010000
#define SYSTEM_MDCCSR_EL0		0b1001100000001000
#define SYSTEM_MDSCR_EL1		0b1000000000010010
#define SYSTEM_MDCR_EL2			0b1110000010001001
#define SYSTEM_MDCR_EL3			0b1111000010011001

#define SYSTEM_OSDLR_EL1		0b1000000010011100
#define SYSTEM_OSDTRRX_EL1		0b1000000000000010
#define SYSTEM_OSDTRTX_EL1		0b1000000000011010
#define SYSTEM_OSECCR_EL1		0b1000000000110010
#define SYSTEM_OSLSR_EL1		0b1000000010001100

#define SYSTEM_PMCCFILTR_EL0	0b1101111101111111
#define SYSTEM_PMCCNTR_EL0		0b1101110011101000
#define SYSTEM_PMCEID0_EL0		0b1101110011100110
#define SYSTEM_PMCEID1_EL0		0b1101110011100111
#define SYSTEM_PMCNTENCLR_EL0	0b1101110011100010
#define SYSTEM_PMCNTENSET_EL0	0b1101110011100001
#define SYSTEM_PMCR_EL0			0b1101110011100000
#define SYSTEM_PMEVCNTR0_EL0	0b1101111101000000
#define SYSTEM_PMEVCNTR1_EL0	0b1101111101000001
#define SYSTEM_PMEVCNTR2_EL0	0b1101111101000010
#define SYSTEM_PMEVCNTR3_EL0	0b1101111101000011
#define SYSTEM_PMEVCNTR4_EL0	0b1101111101000100
#define SYSTEM_PMEVCNTR5_EL0	0b1101111101000101
#define SYSTEM_PMEVTYPER0_EL0	0b1101111101100000
#define SYSTEM_PMEVTYPER1_EL0	0b1101111101100001
#define SYSTEM_PMEVTYPER2_EL0	0b1101111101100010
#define SYSTEM_PMEVTYPER3_EL0	0b1101111101100011
#define SYSTEM_PMEVTYPER4_EL0	0b1101111101100100
#define SYSTEM_PMEVTYPER5_EL0	0b1101111101100101
#define SYSTEM_PMINTENCLR_EL1	0b1100010011110010
#define SYSTEM_PMINTENSET_EL1	0b1100010011110001
#define SYSTEM_PMOVSCLR_EL0		0b1101110011100011
#define SYSTEM_PMOVSSET_EL0		0b1101110011110011
#define SYSTEM_PMSELR_EL0		0b1101110011100101
#define SYSTEM_PMUSERENR_EL0	0b1101110011110000
#define SYSTEM_PMXEVCNTR_EL0	0b1101110011101010
#define SYSTEM_PMXEVTYPER_EL0	0b1101110011101001

/* ARMv8 address translation */
#define SYSTEM_PAR_EL1			0xC3A0
#define SYSTEM_ATS12E0R			0x63C6
#define SYSTEM_ATS12E1R			0x63C4
#define SYSTEM_ATS1E2R			0x63C0
#define SYSTEM_ATS1E3R			0x73C0

/* fault status and fault address */
#define SYSTEM_FAR_EL1			0xC300
#define SYSTEM_FAR_EL2			0xE300
#define SYSTEM_FAR_EL3			0xF300
#define SYSTEM_ESR_EL1			0xC290
#define SYSTEM_ESR_EL2			0xE290
#define SYSTEM_ESR_EL3			0xF290

#define SYSTEM_VBAR_EL1			0b1100011000000000
#define SYSTEM_VBAR_EL2			0b1110011000000000
#define SYSTEM_VBAR_EL3			0b1111011000000000

#define ARMV8_MRS_DSPSR(rt)	(0xd53b4500 | (rt))
#define ARMV8_MSR_DSPSR(rt)	(0xd51b4500 | (rt))
#define ARMV8_MRS_DLR(rt)	(0xd53b4520 | (rt))
#define ARMV8_MSR_DLR(rt)	(0xd51b4520 | (rt))

/* T32 instruction to access coprocessor registers */
#define ARMV8_MCR_T1(cp, crn, opc1, crm, opc2, rt) ARMV4_5_MCR(cp, opc1, rt, crn, crm, opc2)
#define ARMV8_MRC_T1(cp, crn, opc1, crm, opc2, rt) ARMV4_5_MRC(cp, opc1, rt, crn, crm, opc2)

/* T32 instructions to access DSPSR and DLR */
#define ARMV8_MRC_DSPSR(rt) ARMV8_MRC_T1(15, 4, 3, 5, 0, rt)
#define ARMV8_MCR_DSPSR(rt) ARMV8_MCR_T1(15, 4, 3, 5, 0, rt)
#define ARMV8_MRC_DLR(rt)	ARMV8_MRC_T1(15, 4, 3, 5, 1, rt)
#define ARMV8_MCR_DLR(rt)	ARMV8_MCR_T1(15, 4, 3, 5, 1, rt)

#define ARMV8_DCPS1(im)		(0xd4a00001 | (((im) & 0xFFFF) << 5))
#define ARMV8_DCPS2(im)		(0xd4a00002 | (((im) & 0xFFFF) << 5))
#define ARMV8_DCPS3(im)		(0xd4a00003 | (((im) & 0xFFFF) << 5))
#define ARMV8_DCPS(el, im)	(0xd4a00000 | (((im) & 0xFFFF) << 5) | el)
#define ARMV8_DCPS_T1(el)	(0xf78f8000 | el)
#define ARMV8_DRPS		0xd6bf03e0
#define ARMV8_ERET_T1		0xf3de8f00

#define ARMV8_DSB_SY				0xd5033F9F
#define ARMV8_DSB_SY_T1				0xf3bf8f4f
#define ARMV8_ISB				0xd5033fdf
#define ARMV8_ISB_SY_T1				0xf3bf8f6f

#define ARMV8_MRS(system, rt)	(0xd5300000 | ((system) << 5) | (rt))
/* ARM V8 Move to system register. */
#define ARMV8_MSR_GP(system, rt) \
	(0xd5100000 | ((system) << 5) | (rt))
/* ARM V8 Move immediate to process state field. */
#define ARMV8_MSR_IM(op1, crm, op2) \
	(0xd500401f | ((op1) << 16)  | ((crm) << 8) | ((op2) << 5))

#define ARMV8_MRS_T1(r, m1, rd, m) (0xF3E08020 | (r << 20) | (m1 << 16) | (rd << 8) | (m << 4))
#define ARMV8_MRS_XPSR_T1(r, rd) (0xF3EF8000 | (r << 20) | (rd << 8))
#define ARMV8_MSR_GP_T1(r, m1, rd, m) (0xF3808020 | (r << 20) | (m1 << 8) | (rd << 16) | (m << 4))
#define ARMV8_MSR_GP_XPSR_T1(r, rn, mask) (0xF3808000 | (r << 20) | (rn << 16) | (mask << 8))

#define ARMV8_BKPT(im) (0xD4200000 | ((im & 0xffff) << 5))
#define ARMV8_HLT(im) (0x0D4400000 | ((im & 0xffff) << 5))
#define ARMV8_HLT_A1(im) (0xE1000070 | ((im & 0xFFF0) << 4) | (im & 0xF))
#define ARMV8_HLT_T1(im) (0xba80 | (im & 0x3f))

#define ARMV8_MOVFSP_64(rt) ((1 << 31) | 0x11000000 | (0x1f << 5) | (rt))
#define ARMV8_MOVTSP_64(rt) ((1 << 31) | 0x11000000 | (rt << 5) | (0x1F))
#define ARMV8_MOVFSP_32(rt) (0x11000000 | (0x1f << 5) | (rt))
#define ARMV8_MOVTSP_32(rt) (0x11000000 | (rt << 5) | (0x1F))

#define ARMV8_LDRB_IP(rd, rn) (0x38401400 | (rn << 5) | rd)
#define ARMV8_LDRH_IP(rd, rn) (0x78402400 | (rn << 5) | rd)
#define ARMV8_LDRW_IP(rd, rn) (0xb8404400 | (rn << 5) | rd)
#define ARMV8_LDRD_IP(rd, rn) (0xf8408400 | (rn << 5) | rd)

#define ARMV8_LDRB_IP_T3(rd, rn) (0xf8100b01 | (rn << 16) | (rd << 12))
#define ARMV8_LDRH_IP_T3(rd, rn) (0xf8300b02 | (rn << 16) | (rd << 12))
#define ARMV8_LDRW_IP_T3(rd, rn) (0xf8500b04 | (rn << 16) | (rd << 12))

#define ARMV8_STRB_IP(rd, rn) (0x38001400 | (rn << 5) | rd)
#define ARMV8_STRH_IP(rd, rn) (0x78002400 | (rn << 5) | rd)
#define ARMV8_STRW_IP(rd, rn) (0xb8004400 | (rn << 5) | rd)
#define ARMV8_STRD_IP(rd, rn) (0xf8008400 | (rn << 5) | rd)

#define ARMV8_STRB_IP_T3(rd, rn) (0xf8000b01 | (rn << 16) | (rd << 12))
#define ARMV8_STRH_IP_T3(rd, rn) (0xf8200b02 | (rn << 16) | (rd << 12))
#define ARMV8_STRW_IP_T3(rd, rn) (0xf8400b04 | (rn << 16) | (rd << 12))

#define ARMV8_MOV_GPR_VFP(rd, rn, index) (0x4e083c00 | (index << 20) | (rn << 5) | rd)
#define ARMV8_MOV_VFP_GPR(rd, rn, index) (0x4e081c00 | (index << 20) | (rn << 5) | rd)

#define ARMV8_MRS_FPCR(rt)	(0xd53b4400 | (rt))
#define ARMV8_MRS_FPSR(rt)	(0xd53b4420 | (rt))
#define ARMV8_MSR_FPCR(rt)	(0xd51b4400 | (rt))
#define ARMV8_MSR_FPSR(rt)	(0xd51b4420 | (rt))

#define ARMV8_SYS(system, rt) (0xD5080000 | ((system) << 5) | rt)

enum armv8_opcode {
	READ_REG_CTR,
	READ_REG_CLIDR,
	READ_REG_CSSELR,
	READ_REG_CCSIDR,
	WRITE_REG_CSSELR,
	READ_REG_MPIDR,
	READ_REG_DTRRX,
	WRITE_REG_DTRTX,
	WRITE_REG_DSPSR,
	READ_REG_DSPSR,
	ARMV8_OPC_DSB_SY,
	ARMV8_OPC_DCPS,
	ARMV8_OPC_DRPS,
	ARMV8_OPC_ISB_SY,
	ARMV8_OPC_DCCISW,
	ARMV8_OPC_DCCIVAC,
	ARMV8_OPC_ICIVAU,
	ARMV8_OPC_HLT,
	ARMV8_OPC_STRB_IP,
	ARMV8_OPC_STRH_IP,
	ARMV8_OPC_STRW_IP,
	ARMV8_OPC_STRD_IP,
	ARMV8_OPC_LDRB_IP,
	ARMV8_OPC_LDRH_IP,
	ARMV8_OPC_LDRW_IP,
	ARMV8_OPC_LDRD_IP,
	ARMV8_OPC_NUM,
};

extern uint32_t armv8_opcode(struct armv8_common *armv8, enum armv8_opcode);
extern void armv8_select_opcodes(struct armv8_common *armv8, bool state_is_aarch64);

#endif /* OPENOCD_TARGET_ARMV8_OPCODES_H */
